<?php
declare(strict_types=1);

require_once __DIR__ . '/AuthController.php';
require_once __DIR__ . '/../models/sistema/QrPdf.php';
require_once __DIR__ . '/../models/ticketPesos/ticketPesos.php';
require_once __DIR__ . '/../models/sistema/comprasInternas.php';

class PabloController extends AuthController
{
    public function insertarHistorial(){
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['IdUsuario'], $data['action'])) {
            http_response_code(400);
            echo json_encode([
                'status'  => 'error',
                'message' => 'Parámetros incompletos'
            ]);
            return;
        }

        $productora = $data['productora'] ?? null;

        $model = new Historial();

        $resultado = $model->insertaHistorial(
            $data['IdUsuario'],
            $productora,
            $data['action']
        );

        echo json_encode([
            'status'  => 'ok',
            'message' => $resultado
        ]);
    }

    public function consultarHistorial(){
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['IdUsuario'], $data['fechaInicial'], $data['fechaFinal'])) {
            http_response_code(400);
            echo json_encode([
                'status'  => 'error',
                'message' => 'Parámetros incompletos'
            ]);
            return;
        }

        $model = new Historial();

        $historial = $model->consultaHistorial(
            $data['IdUsuario'],
            $data['fechaInicial'],
            $data['fechaFinal']
        );

        echo json_encode([
            'status'  => 'ok',
            'message' => $historial
        ]);
    }

    public function generarQrPdf()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['IdUsuario'], $data['IdEvento'], $data['IdTicket'], $data['llave'])) {
            http_response_code(400);
            echo json_encode([
                'status'  => 'error',
                'message' => 'Parámetros incompletos'
            ]);
            return;
        }

        $model = new QrPdf();

        $resp = $model->crearQrPdf(
            (int)$data['IdUsuario'],
            (int)$data['IdEvento'],
            (int)$data['IdTicket'],
            (string)$data['llave']
        );

        if ($resp['ok'] === true) {
            echo json_encode([
                'status'  => 'ok',
                'message' => 'QR PDF generado correctamente',
                'data'    => $resp
            ]);
            return;
        }

        http_response_code(500);
        echo json_encode([
            'status'  => 'error',
            'message' => $resp['error'] ?? 'No se pudo generar el QR PDF',
            'data'    => null
        ]);
    }

    public function validarQr()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['token'])) {
            http_response_code(400);
            echo json_encode([
                'status'  => 'error',
                'message' => 'Parámetros incompletos'
            ]);
            return;
        }

        $model = new QrPdf();
        $qr = $model->obtenerPorToken((string)$data['token']);

        if (!$qr) {
            http_response_code(404);
            echo json_encode([
                'status'  => 'error',
                'message' => 'Token no existe'
            ]);
            return;
        }

        echo json_encode([
            'status'  => 'ok',
            'message' => [
                'token'    => $qr->token,
                'idEvento' => $qr->idEvento,
                'idTicket' => $qr->idTicket,
                'usado'    => (int)$qr->usado,
                'usado_en' => $qr->usado_en
            ]
        ]);
    }

    public function usarQr()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['token'])) {
            http_response_code(400);
            echo json_encode([
                'status'  => 'error',
                'message' => 'Parámetros incompletos'
            ]);
            return;
        }

        $model = new QrPdf();
        $ok = $model->marcarUsado((string)$data['token']);

        if ($ok) {
            echo json_encode([
                'status'  => 'ok',
                'message' => 'Entrada válida. Ticket marcado como usado.'
            ]);
            return;
        }

        http_response_code(409);
        echo json_encode([
            'status'  => 'error',
            'message' => 'Token inválido o ya fue usado'
        ]);
    }

    public function verCatalogo()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['tipo'])) {
            http_response_code(400);
            echo json_encode(["status" => "error", "message" => "tipo es requerido"]);
            return;
        }

        $tipo = (string)$data['tipo']; // 'cliente' o 'productor'
        $idCategoria = $data['idCategoria'] ?? null;

        $model = new Catalogo();
        $items = $model->listarCatalogo($tipo, $idCategoria);

        echo json_encode(["status" => "ok", "message" => $items]);
    }

    public function canjearProducto()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['usuarioId'], $data['tipo'], $data['productoId'])) {
            http_response_code(400);
            echo json_encode(["status" => "error", "message" => "Parámetros incompletos"]);
            return;
        }

        $cantidad = $data['cantidad'] ?? 1;

        $model = new Catalogo();
        $resp = $model->canjearProducto(
            (int)$data['usuarioId'],
            (string)$data['tipo'],
            (int)$data['productoId'],
            (int)$cantidad
        );

        if ($resp['ok']) {
            echo json_encode(["status" => "ok", "message" => "Canje realizado", "data" => $resp]);
            return;
        }

        http_response_code(409);
        echo json_encode(["status" => "error", "message" => $resp['error'], "data" => $resp]);
    }

    public function procesarTicketPesosVenta()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['ventaId'])) {
            http_response_code(400);
            echo json_encode(["status"=>"error","message"=>"ventaId es requerido"]);
            return;
        }

        $model = new ticketPesos();
        $resp = $model->procesarVentaTicketPesos((int)$data['ventaId']);

        if ($resp['ok']) {
            echo json_encode(["status"=>"ok","message"=>"Ticketpesos procesados","data"=>$resp]);
            return;
        }

        http_response_code(409);
        echo json_encode(["status"=>"error","message"=>$resp['error'],"data"=>$resp]);
    }

    public function historialCanjes()
    {
        $this->validateToken();

        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['usuarioId'])) {
            http_response_code(400);
            echo json_encode(["status" => "error", "message" => "usuarioId es requerido"]);
            return;
        }

        $model = new Catalogo();
        $hist = $model->historialCanjes((int)$data['usuarioId']);

        echo json_encode(["status" => "ok", "message" => $hist]);
    }

    /* =========================================================
     * NUEVOS ENDPOINTS: COMPRAS INTERNAS / INVENTARIO / RECETAS
     * ========================================================= */

    public function ciCrearCatalogo()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'], $data['nombre'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->crearCatalogo((int)$data['idProductora'], (string)$data['nombre'], $data['descripcion'] ?? null);

        echo json_encode(['status' => 'ok', 'message' => 'Catálogo creado', 'data' => $resp]);
    }

    public function ciListarCatalogos()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'idProductora es requerido']);
            return;
        }

        $estado = $data['estado'] ?? 1;

        $model = new ComprasInternas();
        $items = $model->listarCatalogos((int)$data['idProductora'], $estado === null ? null : (int)$estado);

        echo json_encode(['status' => 'ok', 'message' => $items]);
    }

    public function ciCatalogoAgregarProducto()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idCatalogo'], $data['idProducto'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $orden = $data['orden'] ?? 0;

        $model = new ComprasInternas();
        $resp = $model->agregarProductoACatalogo((int)$data['idCatalogo'], (int)$data['idProducto'], (int)$orden);

        echo json_encode(['status' => 'ok', 'message' => 'Producto agregado al catálogo', 'data' => $resp]);
    }

    public function ciCatalogoQuitarProducto()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idCatalogo'], $data['idProducto'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->quitarProductoDeCatalogo((int)$data['idCatalogo'], (int)$data['idProducto']);

        echo json_encode(['status' => 'ok', 'message' => 'Producto quitado del catálogo', 'data' => $resp]);
    }

    public function ciCatalogoVer()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idCatalogo'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'idCatalogo es requerido']);
            return;
        }

        $model = new ComprasInternas();
        $items = $model->listarCatalogoConProductos((int)$data['idCatalogo']);

        echo json_encode(['status' => 'ok', 'message' => $items]);
    }

    public function ciProductoCrear()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'], $data['nombre'], $data['tipo'], $data['unidad'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->crearProducto($data);

        echo json_encode(['status' => 'ok', 'message' => 'Producto creado', 'data' => $resp]);
    }

    public function ciProductosListar()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'idProductora es requerido']);
            return;
        }

        $tipo = $data['tipo'] ?? null; // FINAL / MP / null
        $estado = $data['estado'] ?? 1;

        $model = new ComprasInternas();
        $items = $model->listarProductos((int)$data['idProductora'], $tipo ? (string)$tipo : null, $estado === null ? null : (int)$estado);

        echo json_encode(['status' => 'ok', 'message' => $items]);
    }

    public function ciProductosFinalesPorCategoria()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'idProductora es requerido']);
            return;
        }

        $idCategoria = $data['idCategoria'] ?? null;

        $model = new ComprasInternas();
        $items = $model->listarProductosFinalesPorCategoria((int)$data['idProductora'], $idCategoria !== null ? (int)$idCategoria : null);

        echo json_encode(['status' => 'ok', 'message' => $items]);
    }

    /* =======================
     * CATEGORÍAS (CRUD)
     * ======================= */

    public function ciCategoriaCrear()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'], $data['nombre'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->crearCategoria((int)$data['idProductora'], (string)$data['nombre'], $data['descripcion'] ?? null);

        echo json_encode(['status' => 'ok', 'message' => 'Categoría creada', 'data' => $resp]);
    }

    public function ciCategoriasListar()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'idProductora es requerido']);
            return;
        }

        $estado = $data['estado'] ?? 1;

        $model = new ComprasInternas();
        $items = $model->listarCategorias((int)$data['idProductora'], $estado === null ? null : (int)$estado);

        echo json_encode(['status' => 'ok', 'message' => $items]);
    }

    public function ciCategoriaActualizar()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'], $data['idCategoria'], $data['nombre'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->actualizarCategoria(
            (int)$data['idProductora'],
            (int)$data['idCategoria'],
            (string)$data['nombre'],
            $data['descripcion'] ?? null,
            array_key_exists('estado', $data) ? (int)$data['estado'] : null
        );

        echo json_encode(['status' => 'ok', 'message' => 'Categoría actualizada', 'data' => $resp]);
    }

    public function ciCategoriaEliminar()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'], $data['idCategoria'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->eliminarCategoria((int)$data['idProductora'], (int)$data['idCategoria']);

        echo json_encode(['status' => 'ok', 'message' => 'Categoría eliminada', 'data' => $resp]);
    }

    public function ciProductoCategoriaAsignar()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'], $data['idProducto'], $data['idCategoria'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->asignarCategoriaAProducto((int)$data['idProductora'], (int)$data['idProducto'], (int)$data['idCategoria']);

        if (!($resp['ok'] ?? false)) {
            http_response_code(409);
            echo json_encode(['status' => 'error', 'message' => $resp['error'] ?? 'No se pudo asignar']);
            return;
        }

        echo json_encode(['status' => 'ok', 'message' => 'Categoría asignada', 'data' => $resp]);
    }

    public function ciProductoCategoriaQuitar()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'], $data['idProducto'], $data['idCategoria'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->quitarCategoriaDeProducto((int)$data['idProductora'], (int)$data['idProducto'], (int)$data['idCategoria']);

        echo json_encode(['status' => 'ok', 'message' => 'Categoría quitada', 'data' => $resp]);
    }

    public function ciProductoCategoriasListar()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'], $data['idProducto'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $items = $model->listarCategoriasDeProducto((int)$data['idProductora'], (int)$data['idProducto']);

        echo json_encode(['status' => 'ok', 'message' => $items]);
    }

    /* =======================
     * RECETAS
     * ======================= */

    public function ciRecetaUpsert()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductoFinal'], $data['nombre'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $rend = isset($data['rendimiento']) ? (float)$data['rendimiento'] : 1.0;
        $unidad = $data['unidadRendimiento'] ?? 'UN';

        $model = new ComprasInternas();
        $resp = $model->upsertReceta((int)$data['idProductoFinal'], (string)$data['nombre'], $rend, (string)$unidad);

        if (!($resp['ok'] ?? false)) {
            http_response_code(409);
            echo json_encode(['status' => 'error', 'message' => $resp['error'] ?? 'No se pudo crear receta']);
            return;
        }

        echo json_encode(['status' => 'ok', 'message' => 'Receta guardada', 'data' => $resp]);
    }

    public function ciRecetaSetItems()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idReceta'], $data['items']) || !is_array($data['items'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'idReceta e items son requeridos']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->setRecetaItems((int)$data['idReceta'], (array)$data['items']);

        echo json_encode(['status' => 'ok', 'message' => 'Items de receta guardados', 'data' => $resp]);
    }

    public function ciRecetaObtenerPorProducto()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductoFinal'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'idProductoFinal es requerido']);
            return;
        }

        $model = new ComprasInternas();
        $items = $model->obtenerRecetaCompletaPorProducto((int)$data['idProductoFinal']);

        echo json_encode(['status' => 'ok', 'message' => $items]);
    }

    /* =======================
     * STOCK
     * ======================= */

    public function ciStockMovimiento()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'], $data['idProducto'], $data['tipo'], $data['cantidad'], $data['motivo'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->registrarMovimientoStock($data);

        echo json_encode(['status' => 'ok', 'message' => 'Movimiento registrado', 'data' => $resp]);
    }

    public function ciStockObtener()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'], $data['idProducto'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $stock = $model->obtenerStockActual((int)$data['idProductora'], (int)$data['idProducto']);

        echo json_encode(['status' => 'ok', 'message' => ['stock' => $stock]]);
    }

    public function ciCalcularProducible()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'], $data['idProductoFinal'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->calcularProducible((int)$data['idProductora'], (int)$data['idProductoFinal']);

        if (!($resp['ok'] ?? false)) {
            http_response_code(409);
            echo json_encode(['status' => 'error', 'message' => $resp['error'] ?? 'No se pudo calcular']);
            return;
        }

        echo json_encode(['status' => 'ok', 'message' => $resp]);
    }

    /* =======================
     * VENTAS INTERNAS
     * ======================= */

    public function ciVentaCrear()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idProductora'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'idProductora es requerido']);
            return;
        }

        $IdUsuario = $data['IdUsuario'] ?? null;

        $model = new ComprasInternas();
        $resp = $model->crearVenta((int)$data['idProductora'], $IdUsuario !== null ? (int)$IdUsuario : null);

        echo json_encode(['status' => 'ok', 'message' => 'Venta creada', 'data' => $resp]);
    }

    public function ciVentaAgregarItem()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idVenta'], $data['idProducto'], $data['cantidad'], $data['precioUnitario'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->agregarItemVenta(
            (int)$data['idVenta'],
            (int)$data['idProducto'],
            (float)$data['cantidad'],
            (int)$data['precioUnitario']
        );

        echo json_encode(['status' => 'ok', 'message' => 'Item agregado', 'data' => $resp]);
    }

    public function ciVentaConfirmar()
    {
        $this->validateToken();
        $data = json_decode(file_get_contents("php://input"), true);

        if (!isset($data['idVenta'])) {
            http_response_code(400);
            echo json_encode(['status' => 'error', 'message' => 'idVenta es requerido']);
            return;
        }

        $model = new ComprasInternas();
        $resp = $model->confirmarVenta((int)$data['idVenta']);

        if (!($resp['ok'] ?? false)) {
            http_response_code(409);
            echo json_encode(['status' => 'error', 'message' => $resp['error'] ?? 'No se pudo confirmar']);
            return;
        }

        echo json_encode(['status' => 'ok', 'message' => 'Venta confirmada', 'data' => $resp]);
    }
    
    
    
    public function ciVentaObtener()
{
    $this->validateToken();
    $data = json_decode(file_get_contents("php://input"), true);

    if (!isset($data['idVenta'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'idVenta es requerido']);
        return;
    }

    $model = new ComprasInternas();
    $resp = $model->obtenerVenta((int)$data['idVenta']);

    if (!($resp['ok'] ?? false)) {
        http_response_code(404);
        echo json_encode(['status' => 'error', 'message' => $resp['error'] ?? 'No encontrado']);
        return;
    }

    echo json_encode(['status' => 'ok', 'message' => $resp]);
}

public function ciVentaQuitarItem()
{
    $this->validateToken();
    $data = json_decode(file_get_contents("php://input"), true);

    if (!isset($data['idVenta'], $data['idItem'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'idVenta e idItem son requeridos']);
        return;
    }

    $model = new ComprasInternas();
    $resp = $model->quitarItemVenta((int)$data['idVenta'], (int)$data['idItem']);

    if (!($resp['ok'] ?? false)) {
        http_response_code(409);
        echo json_encode(['status' => 'error', 'message' => $resp['error'] ?? 'No se pudo quitar']);
        return;
    }

    echo json_encode(['status' => 'ok', 'message' => 'Item quitado', 'data' => $resp]);
}

public function ciVentaActualizarItem()
{
    $this->validateToken();
    $data = json_decode(file_get_contents("php://input"), true);

    if (!isset($data['idVenta'], $data['idItem'], $data['cantidad'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'idVenta, idItem y cantidad son requeridos']);
        return;
    }

    $precioUnitario = array_key_exists('precioUnitario', $data) ? (int)$data['precioUnitario'] : null;

    $model = new ComprasInternas();
    $resp = $model->actualizarItemVenta((int)$data['idVenta'], (int)$data['idItem'], (float)$data['cantidad'], $precioUnitario);

    if (!($resp['ok'] ?? false)) {
        http_response_code(409);
        echo json_encode(['status' => 'error', 'message' => $resp['error'] ?? 'No se pudo actualizar']);
        return;
    }

    echo json_encode(['status' => 'ok', 'message' => 'Item actualizado', 'data' => $resp]);
}

public function ciVentaVaciar()
{
    $this->validateToken();
    $data = json_decode(file_get_contents("php://input"), true);

    if (!isset($data['idVenta'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'idVenta es requerido']);
        return;
    }

    $model = new ComprasInternas();
    $resp = $model->vaciarVenta((int)$data['idVenta']);

    if (!($resp['ok'] ?? false)) {
        http_response_code(409);
        echo json_encode(['status' => 'error', 'message' => $resp['error'] ?? 'No se pudo vaciar']);
        return;
    }

    echo json_encode(['status' => 'ok', 'message' => 'Venta vaciada', 'data' => $resp]);
}

public function ciVentaEliminar()
{
    $this->validateToken();
    $data = json_decode(file_get_contents("php://input"), true);

    if (!isset($data['idVenta'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'idVenta es requerido']);
        return;
    }

    $model = new ComprasInternas();
    $resp = $model->eliminarVenta((int)$data['idVenta']);

    if (!($resp['ok'] ?? false)) {
        http_response_code(409);
        echo json_encode(['status' => 'error', 'message' => $resp['error'] ?? 'No se pudo eliminar']);
        return;
    }

    echo json_encode(['status' => 'ok', 'message' => 'Venta eliminada', 'data' => $resp]);
}

public function ciRecetaQuitarItem()
{
    $this->validateToken();
    $data = json_decode(file_get_contents("php://input"), true);

    if (!isset($data['idReceta'], $data['idItem'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'idReceta e idItem son requeridos']);
        return;
    }

    $model = new ComprasInternas();
    $resp = $model->quitarItemReceta((int)$data['idReceta'], (int)$data['idItem']);

    echo json_encode(['status' => 'ok', 'message' => 'Item quitado de receta', 'data' => $resp]);
}

public function ciRecetaActualizarItem()
{
    $this->validateToken();
    $data = json_decode(file_get_contents("php://input"), true);

    if (!isset($data['idReceta'], $data['idItem'], $data['cantidad'], $data['unidad'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'Parámetros incompletos']);
        return;
    }

    $model = new ComprasInternas();
    $resp = $model->actualizarItemReceta((int)$data['idReceta'], (int)$data['idItem'], (float)$data['cantidad'], (string)$data['unidad']);

    if (!($resp['ok'] ?? false)) {
        http_response_code(409);
        echo json_encode(['status' => 'error', 'message' => $resp['error'] ?? 'No se pudo actualizar']);
        return;
    }

    echo json_encode(['status' => 'ok', 'message' => 'Item receta actualizado', 'data' => $resp]);
}

public function ciRecetaListarItems()
{
    $this->validateToken();
    $data = json_decode(file_get_contents("php://input"), true);

    if (!isset($data['idReceta'])) {
        http_response_code(400);
        echo json_encode(['status' => 'error', 'message' => 'idReceta es requerido']);
        return;
    }

    $model = new ComprasInternas();
    $items = $model->listarItemsReceta((int)$data['idReceta']);

    echo json_encode(['status' => 'ok', 'message' => $items]);
}



}
